from flask import Flask, request, jsonify, session
from flask_cors import CORS
import os
import logging
import re

# 1. Setup Logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = Flask(__name__)
app.secret_key = "iie_global_secret_key"

# 2. CORS Configuration
ALLOWED_ORIGINS = [
    "https://impactintl.edu.sa/",

]

CORS(app, 
     resources={r"/*": {"origins": ALLOWED_ORIGINS}},
     supports_credentials=True
)

# 3. Nuclear CORS Injector
@app.after_request
def add_cors_headers(response):
    origin = request.headers.get('Origin')
    if origin in ALLOWED_ORIGINS:
        response.headers['Access-Control-Allow-Origin'] = origin
        response.headers['Access-Control-Allow-Credentials'] = 'true'
        response.headers['Access-Control-Allow-Headers'] = 'Content-Type, Authorization, x-api-key'
        response.headers['Access-Control-Allow-Methods'] = 'GET, POST, OPTIONS'
    return response

# 4. The Info Data (English and Arabic)
IIE_DATA = {
    "en": {
        "summary": (
            "Welcome to Impact International Education (IIE). We provide high-quality education in Saudi Arabia, "
            "combining rigorous programs with modern infrastructure. We offer Undergraduate, Postgraduate, "
            "and Professional programs across Engineering, Business, and Sciences. \n\n"
            "How can I help you? I can provide details on our Programs, Admissions, Contact Info, or Vision."
        ),
        "vision": (
            "Vision: Transform students to lead in a rapidly changing world with knowledge, skills, and adaptability.\n"
            "Mission: Empower students with Critical Thinking, Creativity, and Resilience."
        ),
        "contact": (
            "Contact Us:\n"
            "- Location: Dhahran, Al Basatin, Saudi Arabia\n"
            "- Email: info@iieducation.org\n"
            "- Phone: +966 50 880 1638\n"
            "- Website: https://iieducation.org"
        ),
        "programs": (
            "Our Programs:\n"
            "- Undergraduate: Engineering, CS, Business, Education, Sciences, Arts\n"
            "- Postgraduate: MBA, MPA, MS/PhD, Data Science\n"
            "- Vocational: Certifications and short courses."
        ),
        "admissions": (
            "Enrollment & Admissions:\n"
            "We offer transparent merit-based admissions with scholarships, bursaries, and flexible payment plans available."
        )
    },
    "ar": {
        "summary": (
            "مرحبًا بكم في إمباكت للتعليم الدولي (IIE). نحن نقدم تعليمًا عالي الجودة في المملكة العربية السعودية، "
            "يجمع بين البرامج الصارمة والبنية التحتية الحديثة. نقدم برامج البكالوريوس والدراسات العليا "
            "والبرامج المهنية في الهندسة والأعمال والعلوم. \n\n"
            "كيف يمكنني مساعدتك؟ يمكنني تزويدك بتفاصيل حول البرامج، القبول، معلومات الاتصال، أو رؤيتنا."
        ),
        "vision": (
            "الرؤية: تحويل الطلاب ليصبحوا قادة في عالم سريع التغير متسلحين بالمعرفة والمهارات والقدرة على التكيف.\n"
            "الرسالة: تمكين الطلاب من التفكير النقدي والإبداع والمرونة."
        ),
        "contact": (
            "تواصل معنا:\n"
            "- الموقع: الظهران، البساتين، المملكة العربية السعودية\n"
            "- البريد الإلكتروني: info@iieducation.org\n"
            "- الهاتف: +966 50 880 1638\n"
            "- الموقع الإلكتروني: https://iieducation.org"
        ),
        "programs": (
            "برامجنا:\n"
            "- البكالوريوس: الهندسة، علوم الحاسب، إدارة الأعمال، التعليم، العلوم، الفنون\n"
            "- الدراسات العليا: ماجستير إدارة الأعمال، الإدارة العامة، الماجستير/الدكتوراه، علم البيانات\n"
            "- المهني: الشهادات والدورات القصيرة."
        ),
        "admissions": (
            "القبول والتسجيل:\n"
            "نقدم نظام قبول يتسم بالشفافية والجدارة، مع توفر المنح الدراسية وخطط الدفع المرنة."
        )
    }
}

# 5. Logic: Detect Language -> Match Keyword
def get_static_response(user_msg):
    # Detect if the message contains Arabic characters
    # This checks strictly for Arabic Unicode range
    is_arabic = bool(re.search(r'[\u0600-\u06FF]', user_msg))
    
    msg_lower = user_msg.lower()

    if is_arabic:
        # --- ARABIC LOGIC ---
        data = IIE_DATA["ar"]
        
        if any(x in user_msg for x in ["تواصل", "رقم", "هاتف", "جوال", "ايميل", "بريد", "موقع", "عنوان", "اين"]):
            return data["contact"]
        
        elif any(x in user_msg for x in ["رؤية", "رسالة", "هدف", "قيم", "عن"]):
            return data["vision"]
        
        elif any(x in user_msg for x in ["برنامج", "برامج", "تخصص", "هندسة", "بكالوريوس", "ماجستير", "دراسة"]):
            return data["programs"]
        
        elif any(x in user_msg for x in ["قبول", "تسجيل", "انضمام", "منحة", "رسوم", "دفع", "سعر"]):
            return data["admissions"]
            
        else:
            return data["summary"]

    else:
        # --- ENGLISH LOGIC ---
        data = IIE_DATA["en"]
        
        if any(x in msg_lower for x in ["contact", "email", "phone", "address", "location", "where"]):
            return data["contact"]
        
        elif any(x in msg_lower for x in ["vision", "mission", "values", "goal", "about"]):
            return data["vision"]
        
        elif any(x in msg_lower for x in ["program", "degree", "course", "study", "engineering", "business", "major"]):
            return data["programs"]
        
        elif any(x in msg_lower for x in ["admission", "enroll", "apply", "scholarship", "payment", "cost", "fee"]):
            return data["admissions"]
            
        else:
            return data["summary"]

@app.route('/')
def home():
    return jsonify({"status": "online", "backend": "bilingual-static-bot"})

@app.route('/ask', methods=['POST', 'OPTIONS'])
def ask():
    # Handle Preflight
    if request.method == "OPTIONS":
        return jsonify({"status": "ok"}), 200

    data = request.json
    user_message = data.get("message")
    
    if not user_message:
        return jsonify({"error": "No message provided"}), 400

    # Get History (maintained for compatibility)
    history = session.get('history', [])

    try:
        # --- LOGIC CALL ---
        bot_text = get_static_response(user_message)
        # ------------------

        # Save History
        history.append({"role": "user", "content": user_message})
        history.append({"role": "assistant", "content": bot_text})
        session['history'] = history
        session.modified = True 
        
        return jsonify({"response": bot_text})

    except Exception as e:
        logger.error(f"❌ ERROR: {e}")
        return jsonify({"response": "System Error / خطأ في النظام"}), 500

if __name__ == '__main__':
    port = int(os.environ.get("PORT", 5000))
    app.run(host='0.0.0.0', port=port)